<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\ENewspaper;
use App\Models\NewsLanguage;
use App\Models\NewsLanguageSubscriber;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ENewspaperFrontController extends Controller
{
    public function getENewspaper()
    {
        $theme  = getTheme();
        $title  = __('frontend-labels.enewspapers.title');
        $userId = auth()->user()->id ?? "0";

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage ? collect([$defaultActiveLanguage->id]) : collect();
            }
        }

        // Check limits without incrementing
        $dailyLimitReached        = false;
        $subscriptionLimitReached = false;

        $user         = auth()->user();
        $subscription = $user ? $user->subscription : null;

        if ($subscription) {
            if ($subscription->hasReachedEPaperLimits()) {
                $subscriptionLimitReached = true;
                session(['show_modal' => 'subscription']);
            }
        } else {
            $freeTrialLimit = Setting::where('name', 'free_trial_e_papers_and_magazines_limit')->value('value') ?? 5;
            $freeTrialLimit = (int) $freeTrialLimit;

            $sessionKey   = 'free_tier_e_paper_count';
            $currentCount = session($sessionKey, 0);

            if ($freeTrialLimit !== -1 && $currentCount >= $freeTrialLimit) {
                $dailyLimitReached = true;
                session(['show_modal' => 'daily']);
            }
        }

        $e_newspapers = ENewspaper::with(['channel', 'newsLanguage'])
            ->whereIn('news_language_id', $subscribedLanguageIds)
            ->orderBy('date', 'desc')
            ->where('type', 'paper')
            ->get();

        $socialsettings = Setting::pluck('value', 'name');
        $epapersetting  = $this->getENewsletterSettings();

        $data = [
            'title'                    => $title,
            'e_newspapers'             => $e_newspapers,
            'theme'                    => $theme,
            'dailyLimitReached'        => $dailyLimitReached,
            'subscriptionLimitReached' => $subscriptionLimitReached,
            'socialsettings'           => $socialsettings,
            'epapersetting'            => $epapersetting,
        ];
        return view('front_end.' . $theme . '.pages.e-news-paper', $data);
    }

    private function getENewsletterSettings()
    {
        $settings = Setting::whereIn('name', [
            'enews_paper_image',
            'enews_paper_title',
        ])->pluck('value', 'name');

        return [
            'enewspaper'      => $settings['enews_paper_image'] ?? asset('public/front_end/classic/images/default/newspaper-advertising-service-500x500-1.png'),
            'enewspapertitle' => $settings['enews_paper_title'] ?? 'Newshunt',
        ];
    }
    public function getMagazine()
    {
        $theme  = getTheme();
        $title  = __('frontend-labels.magazines.title');
        $userId = auth()->user()->id ?? "0";

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage ? collect([$defaultActiveLanguage->id]) : collect();
            }
        }

        // Check limits without incrementing
        $dailyLimitReached        = false;
        $subscriptionLimitReached = false;

        $user         = auth()->user();
        $subscription = $user ? $user->subscription : null;

        if ($subscription) {
            if ($subscription->hasReachedEPaperLimits()) {
                $subscriptionLimitReached = true;
                session(['show_modal' => 'subscription']);
            }
        } else {
            $freeTrialLimit = Setting::where('name', 'free_trial_e_papers_and_magazines_limit')->value('value') ?? 5;
            $freeTrialLimit = (int) $freeTrialLimit;

            $sessionKey   = 'free_tier_e_paper_count';
            $currentCount = session($sessionKey, 0);

            if ($freeTrialLimit !== -1 && $currentCount >= $freeTrialLimit) {
                $dailyLimitReached = true;
                session(['show_modal' => 'daily']);
            }
        }

        $e_magazines = ENewspaper::with(['channel', 'newsLanguage'])
            ->whereIn('news_language_id', $subscribedLanguageIds)
            ->orderBy('date', 'desc')
            ->where('type', 'magazine')
            ->get();

        $socialsettings = Setting::pluck('value', 'name');

        $data = [
            'title'                    => $title,
            'e_magazines'              => $e_magazines,
            'theme'                    => $theme,
            'dailyLimitReached'        => $dailyLimitReached,
            'subscriptionLimitReached' => $subscriptionLimitReached,
            'socialsettings'           => $socialsettings,
        ];
        return view('front_end.' . $theme . '.pages.e-news-magazine', $data);
    }

    public function accessPdf(Request $request, $id)
    {
        $userId       = auth()->check() ? auth()->id() : null;
        $user         = auth()->user();
        $subscription = $user ? $user->subscription : null;

        $eNewspaper = ENewspaper::findOrFail($id);

        if ($subscription) {
            if ($subscription->hasReachedEPaperLimits()) {
                session(['show_modal' => 'subscription']);
                session(['subscription_limit_reached' => true]);
                return redirect()->back()->with('error', 'Subscription limit reached');
            }

            $subscription->incrementEPaperCountWithValidation(1);
        } else {
            $freeTrialLimit = Setting::where('name', 'free_trial_e_papers_and_magazines_limit')->value('value') ?? 5;
            $freeTrialLimit = (int) $freeTrialLimit;

            if ($freeTrialLimit !== -1) {
                $sessionKey   = 'free_tier_e_paper_count';
                $currentCount = session($sessionKey, 0);

                if ($currentCount >= $freeTrialLimit) {
                    session(['show_modal' => 'daily']);
                    session(['daily_limit_reached' => true]);
                    return redirect()->back()->with('error', 'Daily limit reached');
                }
                session([$sessionKey => $currentCount + 1]);
            }
        }

        return redirect(asset('storage/' . $eNewspaper->pdf_path));
    }

    public function showPdf($id)
    {
        $e_newspaper = ENewspaper::with('channel')->findOrFail($id);
        
        $appName     = Setting::where('name', 'app_name')->value('value') ?? 'News Portal';
        
        $title                    = $e_newspaper->channel->name;
        $dailyLimitReached        = false;
        $subscriptionLimitReached = false;

        if ($dailyLimitReached || $subscriptionLimitReached) {
            return redirect()->route('e-newspaper.index')
                ->with('daily_limit_reached', $dailyLimitReached)
                ->with('subscription_limit_reached', $subscriptionLimitReached);
        }

        $theme           = session('theme', 'classic');

        return view('front_end.' . $theme . '.pages.pdf-viewer', compact('e_newspaper', 'theme', 'title', 'appName'
        ));
    }
}
